<?php

namespace App\Extensions\Controller;

use Illuminate\Database\Eloquent\Model;

trait ModelExtension
{
    /**
     *  model
     * @var class-string<Model>|Model
     */
    protected  $model;

    protected array $ttlMap = [];


    public function __construct()
    {

        $this->model = $this->resolveModelName();
    }

    public function resolveModelName(): string
    {
        // Get the full class name of the current controller instance
        $target = get_class($this);

        // Remove 'Controller' suffix from the class name
        $name = str_replace('Controller', '', $target);

        // Get the base class name without the namespace
        $model = class_basename($name);

        // Split the class name by namespace separators
        $namespaceArr = explode('\\', $name);

        // Get the part of the namespace before the last element (the controller name)
        $beforeEnd = count($namespaceArr) - 2;

        // Rebuild the namespace to point to the corresponding model
        $name = 'App\\Models\\' . $namespaceArr[$beforeEnd] . '\\' . $model;

        // Ensure the constructed class name is valid and exists
        if (class_exists($name)) {
            return $name;
        }

        // Handle cases where the model might be directly under 'App\Models' without subfolders
        $alternativeName = 'App\\Models\\' . $model;
        if (class_exists($alternativeName)) {
            return $alternativeName;
        }

        // Throw an exception if no valid model class was found
        throw new \Exception("Model class $name or $alternativeName not found.");
    }

    protected function setModel(string $model)
    {
        $this->model = $model;
    }

    public function getModelName()
    {
        return class_basename($this->model);
    }

    public function cacheKey($method)
    {
        return sprintf('%s_%s', $this->resolveModelName(), $method);
    }

    public function getCacheTtl(string $key, ?int $default = null): ?int
    {
        return $this->ttlMap[$key] ?? $default;
    }

    public function findOneBy($id)
    {
        return $this->model::findOrFail($id);
    }
}
