<?php

namespace App\Http\Controllers\Auth;

use App\Contracts\OtpSenderInterface;
use App\Enums\UserTypeEnum;
use App\Http\Controllers\ViewController;
use App\Models\User;
use App\Services\ThirdParty\FakeOtpSender;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Auth;
use App\Services\Model\UserService;
use App\Utils\Otp;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;

class AuthStudentController extends ViewController
{
    public function __construct(
        protected OtpSenderInterface $otpSender,
        protected UserService $userService,
    ) {
    }

    public function register(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255|unique:users',
            'phone' => 'required|string|max:15|unique:users',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator->errors());
        }

        // Generate a token for the user
        $token = Str::random(60);

        // Create the user
        $user = User::create([
            'name' => $request->name,
            'phone' => $request->phone,
            'password' => Hash::make(Str::random(20)),
            'api_token' => $token,
            'type' => UserTypeEnum::STUDENT
        ]);

        $isCreated = $this->userService->setUser($user)->createStudent();

        if (!$isCreated) {
            throw new \Exception("Student not created");
        }

        // Generate OTP
        $otpUtil = new Otp();
        $otp = $otpUtil->makeOtp($user->id);

        $this->otpSender->setTo($request->phone);
        // Send OTP using the FakeOtpSender
        $isSent = $this->otpSender->send($otp);

        if (!$isSent) {
            throw new \Exception("OTP not sent");
        }

        // Redirect with the token
        return $this->redirectWithNotification(
            redirect()->route('public.otp', ['token' => $token])
        );
    }

    public function sendOtp(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'otp' => 'required|string',
            'token' => 'required|string',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator->errors());
        }

        // Check if the token exists in the request and if it's valid
        $user = User::where('api_token', $request->token)->first();

        if (!$user) {
            return back()->withErrors(['token' => 'Invalid token provided.']);
        }

        // Verify the OTP (you need to implement this based on your OTP logic)
        $otpUtil = new Otp();
        $isValidOtp = $otpUtil->checkOtp($user->id, $request->otp);

        if (!$isValidOtp) {
            return back()->withErrors(['otp' => 'Invalid OTP.']);
        }

        // Proceed with OTP validation success (login or activation logic)
        // For example, you can log the user in:
        Auth::login($user);

        return redirect()->route('student.home'); // Redirect to home or a specific page
    }

    public function resendOtp(Request $request, $token)
    {
        $route_redirect_to = $request->route_redirect_to ?? 'public.otp';
        $section = $request->section ?? null;

        $user = User::where('api_token', $token)->first();

        if (!$user) {
            return back()->withErrors(['token' => 'Invalid token provided.']);
        }

        $otpUtil = new Otp();
        $otp = $otpUtil->makeOtp($user->id);

        $this->otpSender->setTo($user->phone);

        $isSent = $this->otpSender->send($otp);

        if (!$isSent) {
            throw new \Exception("OTP not sent");
        }

        $redirect = redirect()->route($route_redirect_to, [
            'token' => $token,
            'status' => 'success'
        ]);


        return $this->redirectWithNotification(
            $redirect->withFragment($section)
        );
    }

    public function login(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'phone' => 'required|string|max:15',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator->errors());
        }

        // Find the user by phone number
        $user = User::where('phone', $request->phone)->first();

        if (!$user) {
            return back()->withErrors(['phone' => 'user with this phone not found']);
        }

        // Generate a token for the user
        $token = Str::random(60);

        // Store the token or send it in a way you see fit
        $user->update(['api_token' => $token]);

        // Generate OTP
        $otpUtil = new Otp();
        $otp = $otpUtil->makeOtp($user->id);

        $this->otpSender->setTo($request->phone);
        // Send OTP using the FakeOtpSender
        $this->otpSender->send($otp);

        // Redirect to OTP page with the token
        return $this->redirectWithNotification(
            redirect()->route('public.otp', ['token' => $token])
        );
    }

    public function logout(Request $request)
    {
        Auth::guard('web')->logout();

        $request->session()->invalidate();

        $request->session()->regenerateToken();

        return redirect('/');
    }

    public function changePhone(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'otp' => 'required|string',
            'phone' => 'required|string',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator->errors());
        }

        $user = auth()->user();

        $otpUtil = new Otp();
        $isValidOtp = $otpUtil->checkOtp($user->id, $request->otp);

        if (!$isValidOtp) {
            return back()->withErrors(['otp' => 'Invalid OTP.']);
        }


        $user->phone = $request->phone;

        $result = $user->save();

        return $this->redirectWithNotification(
            redirect()->route('student.profile'),
            $result ? 'user phone updated' : 'phone update has been failed',
            $result ? 'success' : 'error'
        );
    }
}
