<?php

namespace App\Utils;

use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Str;

class Otp
{
    /**
     * Create a new class instance.
     */
    public function __construct()
    {
        //
    }

    /**
     * Generate and store an OTP (One-Time Password).
     *
     * @param string $userId
     * @return string
     */
    public function makeOtp(string $userId): string
    {
        // Generate a random 6-digit OTP
        $otp = sprintf(
            '%s%s%s%s%s%s',
            rand(0, 9),
            rand(0, 9),
            rand(0, 9),
            rand(0, 9),
            rand(0, 9),
            rand(0, 9),
        );

        // Store OTP in the cache with an expiration time (e.g., 10 minutes)
        Cache::put("otp_{$userId}", $otp, now()->addMinutes(10));

        return $otp;
    }

    /**
     * Flash the OTP for a short period, used for quick retrieval and checking.
     *
     * @param string $userId
     * @param string $otp
     * @param int $minute
     * @return void
     */
    public function flashOtp(string $userId, string $otp, int $minute = 2): void
    {
        // Store OTP in the cache temporarily for quick access (e.g., 2 minutes)
        Cache::put("flash_otp_{$userId}", $otp, now()->addMinutes($minute));
    }

    /**
     * Check if the provided OTP is valid for the given user.
     *
     * @param string $userId
     * @param string $otp
     * @return bool
     */
    public function checkOtp(string $userId, string $otp): bool
    {
        // Retrieve the OTP from the cache (normal or flashed)
        $cachedOtp = Cache::get("otp_{$userId}");
        $flashedOtp = Cache::get("flash_otp_{$userId}");

        // Check if the OTP matches either the stored or flashed OTP
        if ($cachedOtp === $otp || $flashedOtp === $otp) {
            // OTP is valid, remove it from the cache after use
            Cache::forget("otp_{$userId}");
            Cache::forget("flash_otp_{$userId}");
            return true;
        }

        return false;
    }
}
