import DashboardLayout from '@/Layouts/DashboardLayout'
import { Material } from '@/types/models/material'
import { ButtonAdd, ButtonCreate, ButtonDelete, ButtonEdit, CourseForm, CourseSectionForm, LectureForm, LevelSelect, MaterialForm, SubLevelSelect } from '@/Ui'
import { Card, Select } from 'antd'
import { FormEventHandler, useEffect, useState } from 'react'
import { Divider, List, Typography } from 'antd';
import { slugToNormal } from '@/Utils'
import { Course } from '@/types/models/course'
import { CourseSection } from '@/types/models/course_section'
import { Lecture } from '@/types/models/lecture'
import { FormInput, FormUpload, Modal, PrimaryButton } from '@/Components'
import SecondaryButton from '@/Trash/Comps/SecondaryButton'
import { ID } from '@/types/models'
import { useForm } from '@inertiajs/react'
import { Level } from '@/types/models/level'
import { SubLevel } from '@/types/models/sub_level'
import { Teacher } from '@/types/models/teacher'
import { Video } from '@/types/models/video'

type Props = {
    material: Material
    levels: Level[]
    sub_levels: SubLevel[]
    teachers: Teacher[]
    videos: Video[]
}

const MaterialDetails = ({ material, levels, sub_levels,teachers,videos }: Props) => {


    const [openModal, setOpenModal] = useState(false)
    const [sectionModal, setSectionModal] = useState<
        'form_material' | 'delete_material' |
        'form_course' | 'delete_course' |
        'form_courseSection' | 'delete_courseSection' |
        'form_lecture' | 'delete_lecture'
    >()

    const closeModal = () => {
        setOpenModal(false)
    }

    const data = [
        { label: 'العنوان', text: material.title },
        { label: 'الوصف', text: material.description },
        { label: 'الصف', text: material.level.title },
        { label: 'الصف الفرعي', text: material.sub_level?.title },
        { label: 'عدد المناهج', text: material.courses.length.toString() },
    ];

    const [course, setCourse] = useState<Course>()
    const [courseSection, setCourseSection] = useState<CourseSection>()
    const [lecture, setLecture] = useState<Lecture>()



    const handleDeleteMaterial = () => {
        setOpenModal(true)
        setSectionModal('delete_material')
    }
    const DeleteMaterialForm = ({ id }: { id: ID }) => {

        const form = useForm()

        const submit: FormEventHandler = (e) => {
            e.preventDefault()
            form.delete(route('admin.material.deleteOne', { id }), {
                onSuccess: () => setOpenModal(false)
            })
        }

        return <form onSubmit={submit} >
            <p className="my-8 text-md " >هل تريد حقا حذف المادة</p>
            <div className="flex gap-3">
                <PrimaryButton className='bg-red-500' >حذف</PrimaryButton>
                <SecondaryButton>إلغاء</SecondaryButton>
            </div>
        </form>
    }

    const handleEditMaterial = () => {
        setOpenModal(true)
        setSectionModal('form_material')
    }


    const EditMaterialForm = () => {
        const form = useForm(material)

        const submit: FormEventHandler = (e) => {
            e.preventDefault()
            form.put(route('admin.material.updateOne', { id: material.id }), {
                onSuccess: () => setOpenModal(false)
            })
        }

        return <MaterialForm form={form} levels={levels} sub_levels={sub_levels} submit={submit} />
    }

    /** @course */

    const courseForm = useForm(course ? course : {
        id: '',
        material_id: material.id,
        title: '',
        description: '',
        number: material.courses.length + 1,
        is_active: true
    })

    const createCourse: FormEventHandler = (e) => {
        e.preventDefault()
        courseForm.post(route('admin.course.createOne'), {
            onSuccess: () => setOpenModal(false)
        })
    }

    const updateCourse: FormEventHandler = (e) => {
        e.preventDefault()
        courseForm.put(route('admin.course.updateOne', { id: course?.id }), {
            onSuccess: () => setOpenModal(false)
        })
    }

    const handleAddCourse = () => {
        setCourse(undefined)
        setOpenModal(true)
        setSectionModal('form_course')
    }

    const handleUpdateCourse = () => {
        setCourse(course)
        setOpenModal(true)
        setSectionModal('form_course')
    }

    const DeleteCourseForm = ({ id }: { id: ID }) => {

        const form = useForm()

        const submit: FormEventHandler = (e) => {
            e.preventDefault()
            form.delete(route('admin.course.deleteOne', { id }), {
                onSuccess: () => setOpenModal(false)
            })
        }

        return <form onSubmit={submit} >
            <p className="my-8 text-md " >هل تريد حقا حذف المنهج</p>
            <div className="flex gap-3">
                <PrimaryButton className='bg-red-500' >حذف</PrimaryButton>
                <SecondaryButton>إلغاء</SecondaryButton>
            </div>
        </form>
    }

    const handleDeleteCourse = () => {
        setOpenModal(true)
        setSectionModal('delete_course')
    }

    /** @courseSection */

    useEffect(() => {
        if (course) {
            setCourseSection(undefined)
            courseSectionForm.setData('course_id', course.id)
        }

    }, [course])

    const defaultCourseSection: CourseSection = {
        id: '',
        course_id: course?.id ?? '',
        title: '',
        description: '',
        number: (course?.sections?.length || 0) + 1,
        is_active: true,
    }

    const courseSectionForm = useForm(courseSection ?? defaultCourseSection)

    const createCourseSection: FormEventHandler = (e) => {
        e.preventDefault()
        courseSectionForm.post(route('admin.courseSection.createOne'), {
            onSuccess: () => setOpenModal(false)
        })
    }

    const updateCourseSection: FormEventHandler = (e) => {
        e.preventDefault()
        courseSectionForm.put(route('admin.courseSection.updateOne', { id: courseSection?.id }), {
            onSuccess: () => setOpenModal(false)
        })
    }

    const handleAddCourseSection = () => {
        setCourseSection(undefined)
        setOpenModal(true)
        setSectionModal('form_courseSection')
    }

    const handleUpdateCourseSection = () => {
        setCourseSection(courseSection)
        setOpenModal(true)
        setSectionModal('form_courseSection')
    }

    const DeleteCourseSectionForm = ({ id }: { id: ID }) => {

        const form = useForm()

        const submit: FormEventHandler = (e) => {
            e.preventDefault()
            form.delete(route('admin.courseSection.deleteOne', { id }), {
                onSuccess: () => setOpenModal(false)
            })
        }

        return <form onSubmit={submit} >
            <p className="my-8 text-md " >هل تريد حقا حذف المنهج</p>
            <div className="flex gap-3">
                <PrimaryButton className='bg-red-500' >حذف</PrimaryButton>
                <SecondaryButton>إلغاء</SecondaryButton>
            </div>
        </form>
    }

    const handleDeleteCourseSection = () => {
        setOpenModal(true)
        setSectionModal('delete_courseSection')
    }

    /** @lecture */

    useEffect(() => {
        if (course) {
            lectureForm.setData('course_id', course.id)
        }
        if (courseSection){
            lectureForm.setData('course_section_id',courseSection.id)
        }
    }, [course,courseSection])

    const defaultLecture: Lecture = {
        id: '',
        course_id: course?.id ?? '',
        course_section_id: course?.id ?? '',
        title: '',
        description: '',
        number: (course?.sections?.length || 0) + 1,
        is_active: true,
        teacher_id: '',
        video_id: '',
        time: '00:00',
    }

    const lectureForm = useForm(lecture ?? defaultLecture)

    const createLecture: FormEventHandler = (e) => {
        e.preventDefault()
        lectureForm.post(route('admin.lecture.createOne'), {
            onSuccess: () => setOpenModal(false)
        })
    }

    const updateLecture: FormEventHandler = (e) => {
        e.preventDefault()
        lectureForm.put(route('admin.lecture.updateOne', { id: lecture?.id }), {
            onSuccess: () => setOpenModal(false)
        })
    }

    const handleAddLecture = () => {
        setLecture(undefined)
        setOpenModal(true)
        setSectionModal('form_lecture')
    }

    const handleUpdateLecture = () => {
        setLecture(lecture)
        setOpenModal(true)
        setSectionModal('form_lecture')
    }

    const DeleteLectureForm = ({ id }: { id: ID }) => {

        const form = useForm()

        const submit: FormEventHandler = (e) => {
            e.preventDefault()
            form.delete(route('admin.lecture.deleteOne', { id }), {
                onSuccess: () => setOpenModal(false)
            })
        }

        return <form onSubmit={submit} >
            <p className="my-8 text-md " >هل تريد حقا حذف المنهج</p>
            <div className="flex gap-3">
                <PrimaryButton className='bg-red-500' >حذف</PrimaryButton>
                <SecondaryButton>إلغاء</SecondaryButton>
            </div>
        </form>
    }

    const handleDeleteLecture = () => {
        setOpenModal(true)
        setSectionModal('delete_lecture')
    }

    return (
        <DashboardLayout title={material.title} >
            <Modal show={openModal} onClose={closeModal} >
                {sectionModal === 'delete_material' && <DeleteMaterialForm id={material.id} />}
                {sectionModal === 'form_material' && <EditMaterialForm />}
                {sectionModal === 'form_course' && <CourseForm form={courseForm} submit={course ? updateCourse : createCourse} />}
                {(sectionModal === 'delete_course' && course) && <DeleteCourseForm id={course.id} />}
                {sectionModal === 'form_courseSection' && <CourseSectionForm form={courseSectionForm} submit={courseSection ? updateCourseSection : createCourseSection} />}
                {(sectionModal === 'delete_courseSection' && courseSection) && <DeleteCourseSectionForm id={courseSection.id} />}
                {sectionModal === 'form_lecture' && <LectureForm teachers={teachers} videos={videos} form={lectureForm} submit={lecture ? updateLecture : createLecture} />}
                {(sectionModal === 'delete_lecture' && lecture) && <DeleteLectureForm id={lecture.id} />}
            </Modal>
            <div>
                <Divider orientation="left">معلومات المادة</Divider>
                <List
                    className='bg-white'
                    header={<div className='flex justify-end gap-2'>
                        <ButtonEdit onClick={handleEditMaterial} />
                        <ButtonDelete onClick={handleDeleteMaterial} />
                    </div>}
                    bordered
                    dataSource={data}
                    renderItem={(item) => (
                        <List.Item>
                            <List.Item.Meta
                                title={item.label}
                            />
                            <p>
                                {item.text}
                            </p>
                        </List.Item>
                    )}
                />
                <Divider orientation="left">المناهج</Divider>
                <List
                    className='bg-white'
                    bordered
                    header={
                        <div className='flex justify-between items-center gap-3'>
                            <div className='flex flex-1 items-center gap-3'>
                                <label>الكورسات</label>
                                <Select className='w-1/4' onSelect={c => setCourse(material.courses.find(m => m.id == c))} >
                                    {material.courses.map(m => <Select.Option key={m.id} > {m.title} </Select.Option>)}
                                </Select>
                            </div>
                            <div className="flex justify-end gap-2">
                                <ButtonCreate onClick={handleAddCourse} />
                                {course && <ButtonEdit onClick={handleUpdateCourse} />}
                                {course && <ButtonDelete onClick={handleDeleteCourse} />}
                            </div>
                        </div>
                    }
                    dataSource={[
                        { label: 'العنوان', text: course?.title },
                        { label: 'الوصف', text: course?.description },
                        { label: 'عدد الأقسام', text: course?.sections?.length.toString() },
                    ]}
                    renderItem={(courseSelected) => (
                        <List.Item >
                            <List.Item.Meta
                                title={courseSelected.label}
                            />
                            <p>
                                {courseSelected.text}
                            </p>
                        </List.Item>
                    )}
                >


                </List>
                {course && <>
                    <Divider orientation="left">أقسام المنهج</Divider>
                    <List
                        className='bg-white'
                        bordered
                        header={
                            <div className='flex justify-between items-center gap-3'>
                                <div className='flex flex-1 items-center gap-3'>
                                    <label>القسم</label>
                                    <Select className='w-1/3' onSelect={c => setCourseSection(course?.sections?.find(m => m.id == c))} >
                                        {course?.sections?.map(m => <Select.Option key={m.id} > {m.title} </Select.Option>)}
                                    </Select>
                                </div>
                                <div className="flex justify-end gap-2">
                                    <ButtonCreate onClick={handleAddCourseSection} />
                                    {courseSection && <ButtonEdit onClick={handleUpdateCourseSection} />}
                                    {courseSection && <ButtonDelete onClick={handleDeleteCourseSection} />}
                                </div>
                            </div>
                        }
                        dataSource={[
                            { label: 'العنوان', text: courseSection?.title },
                            { label: 'الوصف', text: courseSection?.description },
                            { label: 'عدد الدروس', text: courseSection?.lectures?.length },
                        ]}
                        renderItem={(courseSelected) => (
                            <List.Item >
                                <List.Item.Meta
                                    title={courseSelected.label}
                                />
                                <p>
                                    {courseSelected.text}
                                </p>
                            </List.Item>
                        )}
                    >


                    </List>
                </>}
                {courseSection && <>
                    <Divider orientation="left">الدروس</Divider>
                    <List
                        className='bg-white'
                        bordered
                        header={
                            <div className='flex justify-between items-center gap-3'>
                                <div className='flex flex-1 items-center gap-3'>
                                    <label>الدرس</label>
                                    <Select className='w-1/3' onSelect={c => setLecture(courseSection?.lectures?.find(m => m.id == c))} >
                                        {courseSection.lectures?.map((m, i) => <Select.Option key={i} > {m.title} </Select.Option>)}
                                    </Select>
                                </div>
                                <div className="flex justify-end gap-2">
                                    <ButtonCreate onClick={handleAddLecture}/>
                                    {lecture && <ButtonEdit onClick={handleUpdateLecture} />}
                                    {lecture && <ButtonDelete onClick={handleDeleteLecture}/>}
                                </div>
                            </div>
                        }
                        dataSource={[
                            { label: 'العنوان', text: lecture?.title },
                            { label: 'الوصف', text: lecture?.description },
                            { label: 'الوقت', text: lecture?.time },
                        ]}
                        renderItem={(courseSelected) => (
                            <List.Item >
                                <List.Item.Meta
                                    title={courseSelected.label}
                                />
                                <p>
                                    {courseSelected.text}
                                </p>
                            </List.Item>
                        )}
                    >
                    </List>
                </>}
            </div>
        </DashboardLayout>
    )
}

export default MaterialDetails

