import { Modal, PrimaryButton } from '@/Components'
import { APP_URL } from '@/Constants'
import DashboardLayout from '@/Layouts/DashboardLayout'
import SecondaryButton from '@/Trash/Comps/SecondaryButton'
import { ID } from '@/types/models'
import { Level } from '@/types/models/level'
import { Material } from '@/types/models/material'
import { SubLevel } from '@/types/models/sub_level'
import { ButtonAdd, ButtonDelete, MaterialForm } from '@/Ui'
import { ensureHttpsUrl, removeDuplicates, removeDuplicatesArr } from '@/Utils'
import { Link, useForm } from '@inertiajs/react'
import { Card, Select } from 'antd'
import { FilterIcon } from 'lucide-react'
import { FormEventHandler, useState } from 'react'

type Props = {
    materials: Material[]
    levels: Level[]
    sub_levels: SubLevel[]
}

const Materials = ({ materials, levels, sub_levels }: Props) => {

    const [filteredData, setFilteredData] = useState<Material[]>(materials)

    const [modalOpen, setModalOpen] = useState<boolean>(false)
    const [modelSection, setModelSection] = useState('')

    const [selectedItem, setSelectedItem] = useState<Material>()

    const filter = {
        level: undefined,
        sub_level: undefined,
    }

    const handleOnDelete = (item: Material) => {
        setSelectedItem(item)
        setModalOpen(true)
        setModelSection('delete')
    }

    const handleOnEdit = () => {
        setModalOpen(true)
        setModelSection('add')
    }

    return (
        <DashboardLayout title='المواد الدراسية' headChild={<ButtonAdd onClick={handleOnEdit} />}>
            {(modalOpen && modelSection === 'add') && <DetailForm levels={levels} sub_levels={sub_levels} show={modalOpen} onClose={() => setModalOpen(false)} />}
            {(modalOpen && modelSection === 'delete' && selectedItem) && <DeleteForm item={selectedItem} show={modalOpen} onClose={() => setModalOpen(false)} />}
            <Filter
                levels={removeDuplicates(filteredData.map(m => m.level), 'id')}
                subLevels={removeDuplicates(filteredData.filter(m => m.sub_level !== null).map(m => m.sub_level), 'id')}
            />
            {filteredData.map((item, i) => (
                <MaterialCard key={i} data={item} onDelete={handleOnDelete} />
            ))}
        </DashboardLayout>
    )
}

export default Materials

type DetailProps = { item?: Material, onClose: CallableFunction, show: boolean, levels: Level[], sub_levels: SubLevel[] }

const DetailForm = ({ item, onClose, show, levels, sub_levels }: DetailProps) => {

    const form = useForm(item)

    const submit: FormEventHandler = (e) => {
        e.preventDefault()
        form.post(route('admin.material.createOne'),{
            onSuccess: () => {
                onClose()
            }
        })
    }

    return <Modal show={show} onClose={onClose} >
        <MaterialForm form={form} levels={levels} sub_levels={sub_levels} submit={submit} />
    </Modal>
}

const MaterialCard = ({ data, onDelete }: { data: Material, onDelete: CallableFunction }) => {
    return <Card className='mb-3 ' >
        <div className="flex">
            <Link href={route('admin.materials.detail', { id: data.id })} className="flex-1 flex flex-col gap-3 items-start">
                <span className='bg-gray-800 rounded-md text-xs text-white px-2 py-1' >{data.level.title} ({data.sub_level?.title}) </span>
                <h3 className='text-lg font-semibold' > {data.title} </h3>
                <img className='h-16 ' src={ensureHttpsUrl(data.cover_image, APP_URL)} alt="" />
            </Link>
            <div className="flex">
                <ButtonDelete onClick={() => onDelete(data)} />
            </div>
        </div>
    </Card>
}

const Filter = ({ levels, subLevels }: { levels: Level[], subLevels: SubLevel[] }) => {

    return <div className="p-4 ">
        <h4 className='font-semibold flex text-md text-gray-800'><FilterIcon size={20} /> فلترة </h4>

        <div className="flex gap-3 items-center mt-4 ">
            <Select className='w-1/4' >
                {levels.map((m) => (
                    <Select.Option key={m.id} > {m.title} </Select.Option>
                ))}
            </Select>
            <Select className='w-1/4' >
                {subLevels.map((m) => (
                    <Select.Option key={m.id} > {m.title} </Select.Option>
                ))}
            </Select>
        </div>
    </div>
}

const DeleteForm = ({ item, onClose, show }: { item: Material, onClose: CallableFunction, show: boolean }) => {

    const form = useForm()

    const submit: FormEventHandler = (e) => {
        e.preventDefault()
        form.delete(route('admin.material.deleteOne', { id: item.id }),{
            onSuccess: () => {
                onClose()
            }
        })
    }

    return <Modal show={show} onClose={onClose} >
        <p className="p-4 ">
            هل تريد حقا حذف هذه المادة!
        </p>
        <div className="mt-5 items-center gap-3 flex ">
            <PrimaryButton className='bg-red-500' onClick={submit} >نعم</PrimaryButton>
            <SecondaryButton onClick={() => onClose()} >لا</SecondaryButton>
        </div>
    </Modal>
}
