import { Modal, PrimaryButton } from '@/Components'
import { APP_URL } from '@/Constants'
import DashboardLayout from '@/Layouts/DashboardLayout'
import SecondaryButton from '@/Trash/Comps/SecondaryButton'
import { ID } from '@/types/models'
import { Video } from '@/types/models/video'
import { ButtonAdd, ButtonDelete, ButtonEdit, ButtonPlay, VideoForm } from '@/Ui'
import { ensureHttpsUrl, strShort } from '@/Utils'
import { useForm } from '@inertiajs/react'
import { Card } from 'antd'
import React, { FormEventHandler, useState } from 'react'

type Props = {
    videos: Video[]
}

const Videos = ({ videos }: Props) => {

    const [modalOpen, setModalOpen] = useState<boolean>(false)
    const [modelSection, setModelSection] = useState('')

    const [selectedItem, setSelectedItem] = useState<Video>()

    const onClose = () => {
        setModelSection('')
        setSelectedItem(undefined)
        setModalOpen(false)
    }

    const handleAdd = () => {
        setModalOpen(true)
        setSelectedItem(undefined)
        setModelSection('form')
    }

    const handlePlay = (item: Video) => {
        setModelSection('play')
        setSelectedItem(item)
        setModalOpen(true)
    }

    const handleEdit = (item: Video) => {
        setModelSection('form')
        setSelectedItem(item)
        setModalOpen(true)
    }

    const handleDelete = (item: Video) => {
        setModelSection('delete')
        setSelectedItem(item)
        setModalOpen(true)
    }

    const defaultVideo: Video = {
        id: '',
        file: '',
        subtitle_file: '',
        is_active: true,
    }

    const form = useForm(selectedItem ?? defaultVideo)

    const create: FormEventHandler = (e) => {
        e.preventDefault()
        form.post(route('admin.video.createOne'), {
            onSuccess: () => {
                setModalOpen(false)
            }
        })
    }
    const update: FormEventHandler = (e) => {
        e.preventDefault()
        form.put(route('admin.video.updateOne', selectedItem?.id), {
            onSuccess: () => {
                setModalOpen(false)
            }
        })
    }




    const VideoCard = ({ data }: { data: Video }) => {
        return (
            <div className=" border rounded-md p-2 w-[200px] ">
                <video className='h-[150px] w-full rounded-md' >
                    <source src={ensureHttpsUrl(data.file, APP_URL)} type="video/mp4" />
                    <track src={ensureHttpsUrl(data.subtitle_file, APP_URL)} kind="subtitles" srcLang="en" label="English" />
                </video>
                <p dir='ltr' className='mb-2'>
                    {strShort(data.file.split('/').pop() ?? '',20)}
                </p>
                <div className="flex gap-2">
                    <ButtonDelete onClick={() => handleDelete(data)} />
                    <ButtonPlay onClick={() => handlePlay(data)} />
                    <ButtonEdit onClick={() => handleEdit(data)} />
                </div>
            </div>
        )
    }


    const DeleteForm = ({ id }: { id: ID, }) => {

        const form = useForm()

        const submit: FormEventHandler = (e) => {
            e.preventDefault()
            form.delete(route('admin.video.deleteOne', { id }), {
                onSuccess: () => {
                    onClose()
                }
            })
        }

        return <div >
            <p className="p-4 ">
                هل تريد حقا حذف الفيديو!
            </p>
            <div className="mt-5 items-center gap-3 flex ">
                <PrimaryButton className='bg-red-500' onClick={submit} >نعم</PrimaryButton>
                <SecondaryButton onClick={() => onClose()} >لا</SecondaryButton>
            </div>
        </div>
    }



    return (
        <DashboardLayout title='الفيديوهات' headChild={<ButtonAdd onClick={handleAdd} />} >
            <Modal show={modalOpen} onClose={() => setModalOpen(false)}>
                {(modelSection === 'play' && selectedItem) && <video src={ensureHttpsUrl(selectedItem.file, APP_URL)} controls></video>}
                {modelSection === 'form' && <VideoForm form={form} submit={selectedItem ? update : create} />}
                {(modelSection === 'delete' && selectedItem) && <DeleteForm id={selectedItem?.id} />}
            </Modal>
            <div className="flex gap-3">
                {videos.map((item, i) => (
                    <VideoCard key={i} data={item} />
                ))}
            </div>
        </DashboardLayout>
    )
}

export default Videos

